<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Modules\AdminTeam\App\Models\AdminRole;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Admin extends Authenticatable
{
    use HasFactory;
    protected $guarded = [
        'id', // Prevent the 'id' from being mass assigned
    ];

    const STATUS_ACTIVE = 'enable';

    const STATUS_INACTIVE = 'disable';

    public function roles()
    {
        return $this->belongsToMany(AdminRole::class, 'admin_role_assignments', 'admin_id', 'admin_role_id');
    }

    /**
     * Get cached permissions for this admin
     */
    private function getCachedPermissions()
    {
        $cacheKey = "admin_permissions_{$this->id}";

        return cache()->remember($cacheKey, 3600, function () {
            // Super admin has all permissions
            if ($this->admin_type === 'super_admin') {
                return \Modules\AdminTeam\App\Models\AdminPermissionList::pluck('name')->toArray();
            }

            // Get all permissions through roles
            return $this->roles()->with('permissions')->get()
                ->pluck('permissions')
                ->flatten()
                ->pluck('name')
                ->unique()
                ->toArray();
        });
    }

    /**
     * Clear cached permissions for this admin
     */
    public function clearPermissionCache()
    {
        $cacheKey = "admin_permissions_{$this->id}";
        cache()->forget($cacheKey);
    }

    /**
     * Check if admin has a specific permission
     */
    public function hasPermission($permission)
    {
        $permissions = $this->getCachedPermissions();
        return in_array($permission, $permissions);
    }

    /**
     * Check if admin has any of the given permissions
     */
    public function hasAnyPermission(array $permissions)
    {
        $adminPermissions = $this->getCachedPermissions();
        return !empty(array_intersect($permissions, $adminPermissions));
    }

    /**
     * Check if admin has all of the given permissions
     */
    public function hasAllPermissions(array $permissions)
    {
        $adminPermissions = $this->getCachedPermissions();
        return count(array_intersect($permissions, $adminPermissions)) === count($permissions);
    }

    /**
     * Get cached roles for this admin
     */
    private function getCachedRoles()
    {
        $cacheKey = "admin_roles_{$this->id}";

        return cache()->remember($cacheKey, 3600, function () {
            return $this->roles()->pluck('name')->toArray();
        });
    }

    /**
     * Clear cached roles for this admin
     */
    public function clearRoleCache()
    {
        $cacheKey = "admin_roles_{$this->id}";
        cache()->forget($cacheKey);
    }

    /**
     * Check if admin has a specific role
     */
    public function hasRole($role)
    {
        $roles = $this->getCachedRoles();
        return in_array($role, $roles);
    }

    /**
     * Check if admin has any of the given roles
     */
    public function hasAnyRole(array $roles)
    {
        $adminRoles = $this->getCachedRoles();
        return !empty(array_intersect($roles, $adminRoles));
    }

    /**
     * Get all permissions for this admin
     */
    public function getAllPermissions()
    {
        $permissionNames = $this->getCachedPermissions();

        // Super admin has all permissions
        if ($this->admin_type === 'super_admin') {
            return \Modules\AdminTeam\App\Models\AdminPermissionList::all();
        }

        // Get permission models for cached permission names
        return \Modules\AdminTeam\App\Models\AdminPermissionList::whereIn('name', $permissionNames)->get();
    }

    /**
     * Clear all caches for this admin (permissions and roles)
     */
    public function clearAllCaches()
    {
        $this->clearPermissionCache();
        $this->clearRoleCache();
    }
}
