<?php

namespace Modules\AdminTeam\App\Http\Controllers;

use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use Modules\AdminTeam\App\Models\AdminRole;

class AdminTeamController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        /** @var Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_list_view')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $admins = Admin::with('roles')->latest()->get();

        return view('adminteam::team.index', [
            'admins' => $admins
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        /** @var Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_create')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $roles = AdminRole::where('status', 'active')->get();

        return view('adminteam::team.create', [
            'roles' => $roles
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var Admin|null $adminAuth */
        $adminAuth = auth('admin')->user();
        if(!$adminAuth || !$adminAuth->hasPermission('admin_create')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:admins',
            'password' => 'required|min:4|max:100',
        ], [
            'name.required' => trans('The name field is required.'),
            'email.required' => trans('The email field is required.'),
            'email.email' => trans('The email must be a valid email address.'),
            'email.unique' => trans('The email has already been taken.'),
            'password.required' => trans('The password field is required.'),
            'password.min' => trans('The password must be at least 4 characters.'),
            'password.max' => trans('The password may not be greater than 100 characters.'),
        ]);

        $admin = Admin::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'status' => $request->status ? Admin::STATUS_ACTIVE : Admin::STATUS_INACTIVE,
            'admin_type' => 'team_admin'
        ]);

        // Assign multiple roles to admin
        if ($request->has('role_id') && is_array($request->role_id)) {
            $admin->roles()->attach($request->role_id);
        }

        // Clear admin caches after role assignment
        $admin->clearAllCaches();

        $notification = trans('Created Successfully');
        $notification = array('message' => $notification, 'alert-type' => 'success');

        return redirect()->route('admin.adminteam.index')->with($notification);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        /** @var Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_edit')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $admin = Admin::with('roles')->find($id);

        if ($admin->admin_type == 'super_admin') {
            abort(403, 'Super admin cannot be edited');
        }

        $roles = AdminRole::where('status', 'active')->get();
        $adminRoles = $admin->roles->pluck('id')->toArray();

        return view('adminteam::team.edit', [
            'admin' => $admin,
            'roles' => $roles,
            'adminRoles' => $adminRoles
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        /** @var Admin|null $adminAuth */
        $adminAuth = auth('admin')->user();
        if(!$adminAuth || !$adminAuth->hasPermission('admin_edit')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:admins,email,' . $id,
            'password' => $request->has('change_password_toggle') && $request->change_password_toggle == 'on' ? 'required|min:4|max:100' : 'nullable',
        ], [
            'name.required' => trans('The name field is required.'),
            'email.required' => trans('The email field is required.'),
            'email.email' => trans('The email must be a valid email address.'),
            'email.unique' => trans('The email has already been taken.'),
            'password.required' => trans('The password field is required.'),
            'password.min' => trans('The password must be at least 4 characters.'),
            'password.max' => trans('The password may not be greater than 100 characters.'),
        ]);

        $admin = Admin::find($id);

        if ($admin->admin_type == 'super_admin') {
            abort(403, 'Super admin cannot be edited');
        }

        $admin->name = $request->name;
        $admin->email = $request->email;
        if($request->has('change_password_toggle') && $request->change_password_toggle == 'on'){
            $admin->password = Hash::make($request->password);
        }
        $admin->status = $request->status ? Admin::STATUS_ACTIVE : Admin::STATUS_INACTIVE;
        $admin->save();

        // Update admin roles
        if ($request->has('role_id') && is_array($request->role_id)) {
            $admin->roles()->sync($request->role_id);
        } else {
            // If no roles selected, remove all roles
            $admin->roles()->detach();
        }

        // Clear admin caches after role changes
        $admin->clearAllCaches();

        $notification = trans('Updated Successfully');
        $notification = array('message' => $notification, 'alert-type' => 'success');
        return redirect()->route('admin.adminteam.index')->with($notification);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        /** @var Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_delete')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $admin = Admin::find($id);

        // Detach all roles
        $admin->roles()->detach();

        // Detach all permissions (if direct permissions are used)
        if (method_exists($admin, 'permissions')) {
            $admin->permissions()->detach();
        }

        $admin->delete();
        $admin->clearAllCaches();

        $notification = trans('Deleted Successfully');
        $notification = array('message' => $notification, 'alert-type' => 'success');
        return redirect()->route('admin.adminteam.index')->with($notification);
    }
}
