<?php

namespace Modules\Wallet\App\Http\Controllers\api;

use App\Models\Order;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Wallet\App\Models\Wallet;
use Modules\JobPost\Entities\JobRequest;
use Illuminate\Support\Facades\Validator;

use Modules\Wallet\App\Models\WalletTransaction;
use Modules\PaymentGateway\App\Models\PaymentGateway;

class WalletController extends Controller
{

    public $payment_setting;

    public function __construct()
    {
        $payment_data = PaymentGateway::all();


        $this->payment_setting = array();

        foreach ($payment_data as $data_item) {
            $payment_setting[$data_item->key] = $data_item->value;
        }

        $this->payment_setting  = (object) $payment_setting;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $auth_user = Auth::guard('api')->user();

        $my_wallet = Wallet::where('buyer_id', $auth_user->id)->first();

        if (!$my_wallet) {
            $my_wallet = new Wallet();
            $my_wallet->buyer_id = $auth_user->id;
            $my_wallet->balance = 0.00;
            $my_wallet->save();
        }

        $orders_by_wallet = Order::where('buyer_id', $auth_user->id)->where('payment_method', 'Wallet')->sum('total_amount');


        $jobs_by_wallet = JobRequest::with('job_post:id,regular_price')
            ->where('user_id', $auth_user->id)
            ->where('payment_status', 1)
            ->get();

        $job_post_paid_amount = $jobs_by_wallet->sum(function ($job) {
            return (float) optional($job->job_post)->regular_price;
        });

        $orders_by_wallet = $orders_by_wallet + $job_post_paid_amount;

        $current_balance = $my_wallet->balance - $orders_by_wallet;

        $wallet_transactions = WalletTransaction::where('buyer_id', $auth_user->id)->latest()->get();


        return response()->json([
            'my_wallet' => $my_wallet,
            'current_balance' => $current_balance,
            'orders_by_wallet' => $orders_by_wallet,
            'wallet_transactions' => $wallet_transactions,
            'payment_setting' => $this->payment_setting,
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:1',
            'payment_gateway' => 'required|string|in:Stripe,Paypal,Mollie,Razorpay,Flutterwave,Paystack,Instamojo'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $routes = [
            'Stripe' => 'buyer.wallet-payment.stripe',
            'Paypal' => 'buyer.wallet-payment.paypal',
            'Mollie' => 'buyer.wallet-payment.mollie',
            'Razorpay' => 'buyer.wallet-payment.razorpay',
            'Flutterwave' => 'buyer.wallet-payment.flutterwave',
            'Paystack' => 'buyer.wallet-payment.paystack',
            'Instamojo' => 'buyer.wallet-payment.instamojo',
        ];

        return response()->json([
            'amount' => $request->amount,
            'message' => 'Payment gateway ' . $request->payment_gateway,
            'redirect_url' => route($routes[$request->payment_gateway])
        ], 200);
    }
}
