<?php

namespace Modules\JobPost\Http\Controllers\API\Buyer;

use App\Models\User;
use Auth, Image, File, Str;

use Illuminate\Http\Request;
use Modules\City\Entities\City;
use Illuminate\Routing\Controller;
use Modules\JobPost\Entities\JobPost;

use Modules\Category\Entities\Category;
use Modules\JobPost\Entities\JobRequest;
use Modules\Language\App\Models\Language;

use Illuminate\Contracts\Support\Renderable;

use Modules\JobPost\Entities\JobPostTranslation;
use Modules\JobPost\Http\Requests\JobPostRequest;
use Modules\JobPost\Http\Requests\ApiJobPostRequest;

class JobPostController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        $user = Auth::guard('api')->user();

        $job_posts = JobPost::with('category')->where('user_id', $user->id)->latest()->get();

        $awaiting_job_posts = JobPost::with('category')->where('user_id', $user->id)->where('approved_by_admin', 'pending')->latest()->get();

        $active_job_posts = JobPost::with('category')->where('user_id', $user->id)->where('approved_by_admin', 'approved')->latest()->get();

        $hired_job_posts = JobPost::with('category')->where('user_id', $user->id)->where('approved_by_admin', 'approved')->whereHas('job_applications',function($query) {
            $query->where('status', 'approved');
        })->latest()->get();

        return response()->json([
            'job_posts' => $job_posts,
            'awaiting_job_posts' => $awaiting_job_posts,
            'active_job_posts' => $active_job_posts,
            'hired_job_posts' => $hired_job_posts,
        ]);

    }


    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create(Request $request)
    {

        $categories = Category::where('status', 'enable')->get();

        $cities = City::all();

        $job_types = array(
            'Hourly', 'Daily', 'Monthly', 'Yearly'
        );

        return response()->json([
            'categories' => $categories,
            'cities' => $cities,
            'job_types' => $job_types,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return Renderable
     */
    public function store(ApiJobPostRequest $request)
    {

        $user = Auth::guard('api')->user();

        $job_post = new JobPost();

        if($request->thumb_image){
            $image_name = 'jobpost'.date('-Y-m-d-h-i-s-').rand(999,9999).'.webp';
            $image_name ='uploads/custom-images/'.$image_name;
            Image::make($request->thumb_image)
                ->encode('webp', 80)
                ->save(public_path().'/'.$image_name);
            $job_post->thumb_image = $image_name;
        }

        $job_post->user_id = $user->id;
        $job_post->category_id = $request->category_id;
        $job_post->city_id = $request->city_id;
        $job_post->slug = $request->slug;
        $job_post->regular_price = $request->regular_price;
        $job_post->job_type = $request->job_type;
        $job_post->status = 'enable';
        $job_post->save();

        $languages = Language::all();
        foreach($languages as $language){
            $jobpost_translate = new JobPostTranslation();
            $jobpost_translate->lang_code = $language->lang_code;
            $jobpost_translate->job_post_id = $job_post->id;
            $jobpost_translate->title = $request->title;
            $jobpost_translate->description = $request->description;
            $jobpost_translate->save();
        }


        $notify_message= trans('translate.Created Successfully');

        return response()->json([
            'message' => $notify_message,
            'job_post' => $job_post,
        ]);
    }

    public function jobpost_thumbnail(Request $request){
        $request->validate([
            'job_post_id' => 'required',
            'thumb_image'=> 'required',
        ],[
            'job_post_id.required' => trans('Job id is required'),
            'thumb_image.required' => trans('translate.Image is required'),
        ]);

        $user = Auth::guard('api')->user();

        $job_post = JobPost::where(['user_id' => $user->id, 'id' => $request->job_post_id])->first();

        if(!$job_post){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }


        if($request->thumb_image){
            $old_image = $job_post->thumb_image;
            $image_name = 'jobpost'.date('-Y-m-d-h-i-s-').rand(999,9999).'.webp';
            $image_name ='uploads/custom-images/'.$image_name;
            Image::make($request->thumb_image)
                ->encode('webp', 80)
                ->save(public_path().'/'.$image_name);
            $job_post->thumb_image = $image_name;
            $job_post->save();

            if($old_image){
                if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
            }
        }


        $notify_message= trans('translate.Updated Successfully');

        return response()->json([
            'message' => $notify_message
        ]);


    }


    /**
     * Show the form for editing the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function edit(Request $request, $id)
    {


        $user = Auth::guard('api')->user();

        $job_post = JobPost::where(['user_id' => $user->id, 'id' => $id])->first();

        if(!$job_post){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        $job_post_translate = JobPostTranslation::where(['job_post_id' => $id, 'lang_code' => $request->lang_code])->first();

        $categories = Category::where('status', 'enable')->get();

        $cities = City::all();

        $job_types = array(
            'Hourly', 'Daily', 'Monthly', 'Yearly'
        );

        return response()->json([
            'categories' => $categories,
            'cities' => $cities,
            'job_types' => $job_types,
            'job_post' => $job_post,
            'job_post_translate' => $job_post_translate,
        ]);

    }

    /**
     * Update the specified resource in storage.
     * @param Request $request
     * @param int $id
     * @return Renderable
     */
    public function update(ApiJobPostRequest $request, $id)
    {

        $user = Auth::guard('api')->user();

        $job_post = JobPost::where(['user_id' => $user->id, 'id' => $id])->first();


        if($request->thumb_image){
            $old_image = $job_post->thumb_image;
            $image_name = 'jobpost'.date('-Y-m-d-h-i-s-').rand(999,9999).'.webp';
            $image_name ='uploads/custom-images/'.$image_name;
            Image::make($request->thumb_image)
                ->encode('webp', 80)
                ->save(public_path().'/'.$image_name);
            $job_post->thumb_image = $image_name;
            $job_post->save();

            if($old_image){
                if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
            }
        }
        
        if(!$job_post){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        $jobpost_translate = JobPostTranslation::where(['id' => $request->translate_id, 'job_post_id' => $id])->first();

        if(!$jobpost_translate){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        if($request->lang_code == admin_lang()){

            $job_post->category_id = $request->category_id;
            $job_post->city_id = $request->city_id;
            $job_post->slug = $request->slug;
            $job_post->regular_price = $request->regular_price;
            $job_post->job_type = $request->job_type;
            $job_post->status = 'enable';
            $job_post->save();

        }

        $jobpost_translate->title = $request->title;
        $jobpost_translate->description = $request->description;
        $jobpost_translate->save();

        $notify_message= trans('translate.Updated Successfully');

        return response()->json([
            'message' => $notify_message
        ]);
    }

    /**
     * Remove the specified resource from storage.
     * @param int $id
     * @return Renderable
     */
    public function destroy($id)
    {

        $user = Auth::guard('api')->user();

        $job_post = JobPost::where(['user_id' => $user->id, 'id' => $id])->first();

        if(!$job_post){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        $old_image = $job_post->thumb_image;

        if($old_image){
            if(File::exists(public_path().'/'.$old_image))unlink(public_path().'/'.$old_image);
        }

        JobPostTranslation::where('job_post_id',$id)->delete();
        JobRequest::where('job_post_id',$id)->delete();

        $job_post->delete();

        $notify_message=  trans('translate.Delete Successfully');

        return response()->json([
            'message' => $notify_message
        ]);
    }

    public function assign_language($lang_code){
        $jobpost_translates = JobPostTranslation::where('lang_code', admin_lang())->get();
        foreach($jobpost_translates as $jobpost_translate){
            $translate = new JobPostTranslation();
            $translate->job_post_id = $jobpost_translate->job_post_id;
            $translate->lang_code = $lang_code;
            $translate->title = $jobpost_translate->title;
            $translate->description = $jobpost_translate->description;
            $translate->save();
        }
    }

    public function job_post_applicants($id){

        $user = Auth::guard('api')->user();

        $job_post = JobPost::where(['user_id' => $user->id, 'id' => $id])->first();

        if(!$job_post){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        $job_requests = JobRequest::with('seller')->where('job_post_id', $id)->latest()->get();

        return response()->json([
            'job_requests' => $job_requests
        ]);

    }

    public function job_application_approval($id){

        $user = Auth::guard('api')->user();

        $job_request = JobRequest::where(['user_id' => $user->id, 'id' => $id])->first();

        if(!$job_request){
            return response()->json([
                'message' => trans('Not Found')
            ], 403);
        }

        $approval_check = JobRequest::where('job_post_id', $job_request->job_post_id)->where('status', 'approved')->count();

        if($approval_check == 0){
            $job_request = JobRequest::findOrFail($id);
            $job_request->status = 'approved';
            $job_request->save();

            JobRequest::where('job_post_id', $job_request->job_post_id)->where('id', '!=', $id)->update(['status' => 'rejected']);

            $notify_message = trans('translate.Job assigned successfully');
            return response()->json([
                'message' => $notify_message
            ], 403);

        }else{
            $notify_message = trans('translate.Job already has assigned, so you can not assign again');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }

    }

}
