<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Admin;

class ClearAdminCaches extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'admin:clear-caches {--admin= : Clear cache for specific admin ID}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clear admin permission and role caches';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $adminId = $this->option('admin');

        if ($adminId) {
            // Clear cache for specific admin
            $admin = Admin::find($adminId);
            if ($admin) {
                $admin->clearAllCaches();
                $this->info("Cache cleared for admin: {$admin->name} (ID: {$adminId})");
            } else {
                $this->error("Admin with ID {$adminId} not found.");
            }
        } else {
            // Clear all admin caches
            $this->clearAllAdminCaches();
            $this->info('All admin caches cleared successfully.');
        }
    }

    /**
     * Clear all admin caches
     */
    private function clearAllAdminCaches()
    {
        // More efficient: Clear cache patterns
        $this->clearCachePattern('admin_permissions_*');
        $this->clearCachePattern('admin_roles_*');
    }

    /**
     * Clear cache by pattern
     */
    private function clearCachePattern($pattern)
    {
        try {
            // For Redis
            if (config('cache.default') === 'redis') {
                $redis = app('redis');
                $keys = $redis->keys($pattern);
                if (!empty($keys)) {
                    $redis->del($keys);
                    $this->line("Cleared " . count($keys) . " cache keys matching pattern: {$pattern}");
                }
            } else {
                // Fallback: Get all admins and clear individual caches
                $admins = Admin::pluck('id');
                foreach ($admins as $adminId) {
                    cache()->forget(str_replace('*', $adminId, $pattern));
                }
                $this->line("Cleared caches for " . $admins->count() . " admins");
            }
        } catch (\Exception $e) {
            // Fallback: Get all admins and clear individual caches
            $admins = Admin::pluck('id');
            foreach ($admins as $adminId) {
                cache()->forget(str_replace('*', $adminId, $pattern));
            }
            $this->line("Cleared caches for " . $admins->count() . " admins (fallback method)");
        }
    }
}
