<?php

namespace Modules\AdminTeam\App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use App\Models\Admin;
use Modules\AdminTeam\App\Models\AdminRole;
use Modules\AdminTeam\App\Models\AdminPermissionList;
class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_role_view')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $roles = AdminRole::latest()->get();

        return view('adminteam::roles.index', ['roles' => $roles]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_role_create')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $permissions = AdminPermissionList::where('is_group', true)
            ->with(['children' => function($query) {
                $query->where('is_group', false);
            }])
            ->get();

            // return $permissions;

        return view('adminteam::roles.create', compact('permissions'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_role_create')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:admin_roles',
            'display_name' => 'required|string|max:255|unique:admin_roles',
            'description' => 'nullable|string',
        ], [
            'name.required' => trans('Name is required'),
            'display_name.required' => trans('Display name is required'),
            'name.unique' => trans('Name already exists'),
            'display_name.unique' => trans('Display name already exists'),
        ]);

        $request['status'] = $request->status ? 'active' : 'inactive';



        $role = AdminRole::create($request->all());

        // Assign permissions to the role
        if ($request->has('permissions')) {
            $permissions = [];

            // Add group permissions
            if (isset($request->permissions['groups'])) {
                $permissions = array_merge($permissions, $request->permissions['groups']);
            }

            // Add child permissions
            if (isset($request->permissions['children'])) {
                $permissions = array_merge($permissions, $request->permissions['children']);
            }

            // Remove duplicates and assign
            $permissions = array_unique($permissions);
            $role->permissions()->attach($permissions);
        }

        // Clear all admin caches since role permissions changed
        $this->clearAllAdminCaches();

        $notify_message = trans('Created successfully');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->route('admin.roles.index')->with($notify_message);
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_role_edit')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $role = AdminRole::findOrFail($id);

        if ($role->name == 'super_admin') {
            abort(403, 'Super admin role cannot be edited');
        }

        // Get all permission groups with their children
        $permissions = AdminPermissionList::where('is_group', true)
            ->with(['children' => function($query) {
                $query->where('is_group', false);
            }])
            ->get();

        // Get current role permissions
        $rolePermissions = $role->permissions()->pluck('admin_permission_id')->toArray();

        return view('adminteam::roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        /** @var \App\Models\Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin || !$admin->hasPermission('admin_role_edit')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $request->validate([
            // 'name' => 'required|string|max:255|unique:admin_roles,name,' . $id,
            'display_name' => 'required|string|max:255|unique:admin_roles,display_name,' . $id,
            'description' => 'nullable|string',
        ], [
            'name.required' => trans('Name is required'),
            'display_name.required' => trans('Display name is required'),
            'name.unique' => trans('Name already exists'),
            'display_name.unique' => trans('Display name already exists'),
        ]);

        $request['status'] = $request->status ? 'active' : 'inactive';

        $role = AdminRole::findOrFail($id);

        if ($role->name == 'super_admin') {
            abort(403, 'Super admin role cannot be edited');
        }

        $role->update($request->all());

        // Update permissions for the role
        if ($request->has('permissions')) {
            $permissions = [];

            // Add group permissions
            if (isset($request->permissions['groups'])) {
                $permissions = array_merge($permissions, $request->permissions['groups']);
            }

            // Add child permissions
            if (isset($request->permissions['children'])) {
                $permissions = array_merge($permissions, $request->permissions['children']);
            }

            // Remove duplicates and sync (this will replace existing permissions)
            $permissions = array_unique($permissions);
            $role->permissions()->sync($permissions);
        } else {
            // If no permissions selected, remove all permissions
            $role->permissions()->detach();
        }

        // Clear all admin caches since role permissions changed
        $this->clearAllAdminCaches();

        $notify_message = trans('Updated successfully');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->route('admin.roles.index')->with($notify_message);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        /** @var Admin|null $admin */
        $admin = auth('admin')->user();

        if(!$admin->hasPermission('admin_role_delete')){
            abort(403, 'You do not have permission to access this resource.');
        }

        $role = AdminRole::findOrFail($id);

        // Check if any admins are assigned to this role
        if ($role->admins()->count() > 0) {
            $notify_message = trans('Cannot delete: Role is assigned to one or more admins.');
            $notify_message = array('message' => $notify_message, 'alert-type' => 'error');
            return redirect()->route('admin.roles.index')->with($notify_message);
        }

        $role->permissions()->detach();
        $role->delete();

        $notify_message = trans('Deleted successfully');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->route('admin.roles.index')->with($notify_message);
    }

    /**
     * Clear all admin caches when role permissions change
     */
    private function clearAllAdminCaches()
    {
        // More efficient: Clear cache patterns instead of individual admin caches
        $this->clearCachePattern('admin_permissions_*');
        $this->clearCachePattern('admin_roles_*');
    }

    /**
     * Clear cache by pattern (works with Redis, Memcached, etc.)
     */
    private function clearCachePattern($pattern)
    {
        try {
            // For Redis
            if (config('cache.default') === 'redis') {
                $redis = app('redis');
                $keys = $redis->keys($pattern);
                if (!empty($keys)) {
                    $redis->del($keys);
                }
            } else {
                // Fallback: Get all admins and clear individual caches
                $admins = \App\Models\Admin::pluck('id');
                foreach ($admins as $adminId) {
                    cache()->forget(str_replace('*', $adminId, $pattern));
                }
            }
        } catch (\Exception $e) {
            // Fallback: Get all admins and clear individual caches
            $admins = \App\Models\Admin::pluck('id');
            foreach ($admins as $adminId) {
                cache()->forget(str_replace('*', $adminId, $pattern));
            }
        }
    }
}
