<?php

namespace App\Http\Controllers\API;

use Exception;

use Stripe\Price;
use App\Models\Order;
use Razorpay\Api\Api;
use Stripe, Auth, Session;
use Illuminate\Http\Request;
use Mollie\Laravel\Facades\Mollie;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Modules\Listing\Entities\Listing;
use Modules\Wallet\App\Models\Wallet;

use Modules\JobPost\Entities\JobRequest;
use Modules\Currency\App\Models\Currency;
use Stripe\Checkout\Session as StripSession;
use Modules\Listing\App\Models\ListingPackage;
use Modules\GlobalSetting\App\Models\GlobalSetting;
use Srmklive\PayPal\Services\PayPal as PayPalClient;
use Modules\PaymentGateway\App\Models\PaymentGateway;

class PaymentController extends Controller
{

    public $payment_setting;

    public function __construct()
    {
        $payment_data = PaymentGateway::all();


        $this->payment_setting = array();

        foreach ($payment_data as $data_item) {
            $payment_setting[$data_item->key] = $data_item->value;
        }

        $this->payment_setting  = (object) $payment_setting;
    }

    public function index(Request $request, $service_package_id, $package_name)
    {

        $service_package = ListingPackage::find($service_package_id);

        if (!$service_package) {
            return response()->json([
                'message' => trans('Package Not Found')
            ], 403);
        }

        if ($package_name == 'Basic' || $package_name == 'Standard' || $package_name == 'Premium') {

            $auth_user = Auth::guard('api')->user();

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->first();

            if (!$service) {
                return response()->json([
                    'message' => trans('Service Not Found')
                ], 403);
            }

            $payable_amount = 0.00;
            if ($package_name == 'Basic') {
                $payable_amount = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $payable_amount = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $payable_amount = $service_package->premium_price;
            }

            $razorpay_currency = Currency::findOrFail($this->payment_setting->razorpay_currency_id);
            $flutterwave_currency = Currency::findOrFail($this->payment_setting->flutterwave_currency_id);
            $paystack_currency = Currency::findOrFail($this->payment_setting->paystack_currency_id);

            return response()->json([
                'service_package' => $service_package,
                'service' => $service,
                'package_name' => $package_name,
                'payable_amount' => $payable_amount,
                'payment_setting' => $this->payment_setting,
                'razorpay_currency' => $razorpay_currency,
                'flutterwave_currency' => $flutterwave_currency,
                'paystack_currency' => $paystack_currency,
            ]);
        } else {
            $notify_message = trans('translate.Something went wrong');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }
    }

    public function wallet_payment(Request $request, $service_package_id, $package_name)
    {

        if (env('APP_MODE') == 'DEMO') {
            $notify_message = trans('translate.This Is Demo Version. You Can Not Change Anything');
            $notify_message = array('message' => $notify_message, 'alert-type' => 'error');
            return redirect()->back()->with($notify_message);
        }

        $auth_user = Auth::guard('api')->user();

        $service_package = ListingPackage::findOrFail($service_package_id);
        $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

        $package_main_price = 0.00;
        if ($package_name == 'Basic') {
            $package_main_price = $service_package->basic_price;
        } elseif ($package_name == 'Standard') {
            $package_main_price = $service_package->standard_price;
        } elseif ($package_name == 'Premium') {
            $package_main_price = $service_package->premium_price;
        }

        $my_wallet = Wallet::where('buyer_id', $auth_user->id)->first();

        if (!$my_wallet) {
            $my_wallet = new Wallet();
            $my_wallet->buyer_id = $auth_user->id;
            $my_wallet->balance = 0.00;
            $my_wallet->save();
        }

        $orders_by_wallet = Order::where('buyer_id', $auth_user->id)->where('payment_method', 'Wallet')->sum('total_amount');

        $jobs_by_wallet = JobRequest::with('job_post:id,regular_price')
            ->where('user_id', $auth_user->id)
            ->where('payment_status', 1)
            ->get();

        $job_post_paid_amount = $jobs_by_wallet->sum(function ($job) {
            return (float) optional($job->job_post)->regular_price;
        });

        $orders_by_wallet = $orders_by_wallet + $job_post_paid_amount;

        $current_balance = $my_wallet->balance - $orders_by_wallet;

        if ($current_balance < $package_main_price) {
            $notify_message = trans('translate.Do not have enough balance to your wallet');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }

        $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Wallet', 'success', 'wallet');

        $notify_message = trans('translate.Your payment has been made successful. Thanks for your new purchase');
        return response()->json(['message' => $notify_message]);
    }


    public function stirpe_payment(Request $request, $service_package_id, $package_name)
    {

        $rules = [
            'card_number' => 'required',
            'month' => 'required',
            'year' => 'required',
            'cvc' => 'required',
        ];
        $customMessages = [
            'card_number.required' => trans('translate.Card number is required'),
            'year.required' => trans('translate.Year is required'),
            'month.required' => trans('translate.Month is required'),
            'cvc.required' => trans('translate.CVC is required'),
        ];
        $this->validate($request, $rules, $customMessages);

        $auth_user = Auth::guard('api')->user();

        $service_package = ListingPackage::find($service_package_id);

        if (!$service_package) {
            return response()->json([
                'message' => trans('Package Not Found')
            ], 403);
        }

        $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->first();

        if (!$service) {
            return response()->json([
                'message' => trans('Service Not Found')
            ], 403);
        }

        $package_main_price = 0.00;
        if ($package_name == 'Basic') {
            $package_main_price = $service_package->basic_price;
        } elseif ($package_name == 'Standard') {
            $package_main_price = $service_package->standard_price;
        } elseif ($package_name == 'Premium') {
            $package_main_price = $service_package->premium_price;
        }

        $stripe_currency = Currency::findOrFail($this->payment_setting->stripe_currency_id);

        $payable_amount = round($package_main_price * $stripe_currency->currency_rate, 2);

        Stripe\Stripe::setApiKey($this->payment_setting->stripe_secret);


        try {
            $token = Stripe\Token::create([
                'card' => [
                    'number' => $request->card_number,
                    'exp_month' => $request->month,
                    'exp_year' => $request->year,
                    'cvc' => $request->cvc,
                ],
            ]);
        } catch (Exception $ex) {
            Log::info($ex->getMessage());
            return response()->json(['message' => 'Please provide valid card information'], 403);
        }


        if (!isset($token['id'])) {
            return response()->json(['message' => 'Payment faild'], 403);
        }

        try {


            $result = Stripe\Charge::create([
                'card' => $token['id'],
                'currency' => $stripe_currency->currency_code,
                'amount' => $payable_amount * 100,
                'description' => env('APP_NAME'),
            ]);

            if ($result['status'] != 'succeeded') {
                return response()->json(['message' => 'Payment faild'], 403);
            }
        } catch (Exception $ex) {
            Log::info($ex->getMessage());
            $notify_message = trans('translate.Something went wrong, please try again');
            return response()->json(['message' => $notify_message], 403);
        }

        $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Stripe', 'success', $result->balance_transaction);

        $notify_message = trans('translate.Your payment has been made successful. Thanks for your new purchase');
        return response()->json(['message' => $notify_message]);
    }



    public function bank_payment(Request $request, $service_package_id, $package_name)
    {

        $request->validate([
            'tnx_info' => 'required|max:255'
        ], [
            'tnx_info.required' => trans('translate.Transaction field is required')
        ]);

        $auth_user = Auth::guard('api')->user();

        $service_package = ListingPackage::findOrFail($service_package_id);
        $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

        $package_main_price = 0.00;
        if ($package_name == 'Basic') {
            $package_main_price = $service_package->basic_price;
        } elseif ($package_name == 'Standard') {
            $package_main_price = $service_package->standard_price;
        } elseif ($package_name == 'Premium') {
            $package_main_price = $service_package->premium_price;
        }

        $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Bank Payment', 'pending', $request->tnx_info);

        $notify_message = trans('translate.Your payment has been made. please wait for admin payment approval');
        return response()->json(['message' => $notify_message]);
    }


    public function paypal_webview(Request $request, $service_package_id, $package_name)
    {

        $auth_user = Auth::guard('api')->user();

        $service_package = ListingPackage::findOrFail($service_package_id);
        $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

        $package_main_price = 0.00;
        if ($package_name == 'Basic') {
            $package_main_price = $service_package->basic_price;
        } elseif ($package_name == 'Standard') {
            $package_main_price = $service_package->standard_price;
        } elseif ($package_name == 'Premium') {
            $package_main_price = $service_package->premium_price;
        }

        $paypal_currency = Currency::findOrFail($this->payment_setting->paypal_currency_id);

        $payable_amount = round($package_main_price * $paypal_currency->currency_rate);

        config(['paypal.mode' => $this->payment_setting->paypal_account_mode]);

        if ($this->payment_setting->paypal_account_mode == 'sandbox') {
            config(['paypal.sandbox.client_id' => $this->payment_setting->paypal_client_id]);
            config(['paypal.sandbox.client_secret' => $this->payment_setting->paypal_secret_key]);
        } else {
            config(['paypal.live.client_id' => $this->payment_setting->paypal_client_id]);
            config(['paypal.live.client_secret' => $this->payment_setting->paypal_secret_key]);
            config(['paypal.live.app_id' => 'APP-80W284485P519543T']);
        }

        $provider = new PayPalClient;
        $provider->setApiCredentials(config('paypal'));
        $paypalToken = $provider->getAccessToken();
        $response = $provider->createOrder([
            "intent" => "CAPTURE",
            "application_context" => [
                "return_url" => route('payment.paypal-webview-success'),
                "cancel_url" => route('webview-faild-payment'),
            ],
            "purchase_units" => [
                0 => [
                    "amount" => [
                        "currency_code" => $paypal_currency->currency_code,
                        "value" => $payable_amount
                    ]
                ]
            ]
        ]);

        Session::put('service_package_id', $service_package_id);
        Session::put('package_name', $package_name);
        Session::put('auth_user', $auth_user);


        if (isset($response['id']) && $response['id'] != null) {

            // redirect to approve href
            foreach ($response['links'] as $links) {
                if ($links['rel'] == 'approve') {
                    return redirect()->away($links['href']);
                }
            }

            $notification = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notification]);
        } else {
            $notification = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notification]);
        }
    }



    public function paypal_webview_success(Request $request)
    {


        $paypal_currency = Currency::findOrFail($this->payment_setting->paypal_currency_id);

        config(['paypal.mode' => $this->payment_setting->paypal_account_mode]);

        if ($this->payment_setting->paypal_account_mode == 'sandbox') {
            config(['paypal.sandbox.client_id' => $this->payment_setting->paypal_client_id]);
            config(['paypal.sandbox.client_secret' => $this->payment_setting->paypal_secret_key]);
        } else {
            config(['paypal.live.client_id' => $this->payment_setting->paypal_client_id]);
            config(['paypal.live.client_secret' => $this->payment_setting->paypal_secret_key]);
            config(['paypal.live.app_id' => 'APP-80W284485P519543T']);
        }

        $provider = new PayPalClient;
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();
        $response = $provider->capturePaymentOrder($request['token']);
        if (isset($response['status']) && $response['status'] == 'COMPLETED') {

            $service_package_id = Session::get('service_package_id');

            $package_name = Session::get('package_name');

            $auth_user = Session::get('auth_user');

            $service_package = ListingPackage::findOrFail($service_package_id);

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

            $package_main_price = 0.00;
            if ($package_name == 'Basic') {
                $package_main_price = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $package_main_price = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $package_main_price = $service_package->premium_price;
            }


            $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Paypal', 'success', $request->PayerID);

            $notification = trans('translate.Your payment has been made successful. Thanks for your new purchase');
            return response()->json(['status' => 'success', 'message' => $notification]);
        } else {
            $notification = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notification]);
        }
    }

    public function paypal_faild_payment(Request $request)
    {

        $service_package_id = Session::get('service_package_id');

        $package_name = Session::get('package_name');

        $notify_message = trans('translate.Something went wrong, please try again');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'error');
        return redirect()->route('payment.pay', ['service_package_id' => $service_package_id, 'package_name' => $package_name])->with($notify_message);
    }

    public function razorpay_webview(Request $request, $service_package_id, $package_name)
    {


        if (env('APP_MODE') == 'DEMO') {
            $notification = trans('user_validation.This Is Demo Version. You Can Not Change Anything');
            return response()->json(['message' => $notification], 403);
        }

        $user = Auth::guard('api')->user();
        if (!$user) {
            return response()->json(['message' => 'User Not Found'], 403);
        }

        $service_package = ListingPackage::find($service_package_id);

        if (!$service_package) {
            return response()->json([
                'message' => trans('Package Not Found')
            ], 403);
        }

        if ($package_name == 'Basic' || $package_name == 'Standard' || $package_name == 'Premium') {

            $auth_user = Auth::guard('api')->user();

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->first();

            if (!$service) {
                return response()->json([
                    'message' => trans('Service Not Found')
                ], 403);
            }

            $payable_amount = 0.00;
            if ($package_name == 'Basic') {
                $payable_amount = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $payable_amount = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $payable_amount = $service_package->premium_price;
            }

            $razorpay_currency = Currency::findOrFail($this->payment_setting->razorpay_currency_id);
        } else {
            $notify_message = trans('translate.Something went wrong');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }

        Session::put('auth_user', $user);
        $payment_setting = $this->payment_setting;
        $setting = GlobalSetting::first();

        return view('webview.razorpay_webview', compact('user', 'payable_amount', 'setting', 'payment_setting', 'service_package', 'package_name', 'razorpay_currency'));
    }

    public function razorpay_webview_success(Request $request, $service_package_id, $package_name)
    {

        $input = $request->all();
        $api = new Api($this->payment_setting->razorpay_key, $this->payment_setting->razorpay_secret);
        $payment = $api->payment->fetch($input['razorpay_payment_id']);
        if (count($input)  && !empty($input['razorpay_payment_id'])) {

            try {
                $response = $api->payment->fetch($input['razorpay_payment_id'])->capture(array('amount' => $payment['amount']));
                $payId = $response->id;
                $auth_user = Session::get('auth_user');

                $service_package = ListingPackage::findOrFail($service_package_id);

                $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

                $package_main_price = 0.00;
                if ($package_name == 'Basic') {
                    $package_main_price = $service_package->basic_price;
                } elseif ($package_name == 'Standard') {
                    $package_main_price = $service_package->standard_price;
                } elseif ($package_name == 'Premium') {
                    $package_main_price = $service_package->premium_price;
                }

                $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Razorpay', 'success', $payId);

                $notification = trans('translate.Your payment has been made successful. Thanks for your new purchase');
                return response()->json(['status' => 'success', 'message' => $notification]);
            } catch (Exception $e) {
                $notification = trans('translate.Something went wrong, please try again');
                return response()->json(['status' => 'faild', 'message' => $notification]);
            }
        } else {
            $notification = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notification]);
        }
    }


    public function flutterwave_webview(Request $request, $service_package_id, $package_name)
    {

        if (env('APP_MODE') == 'DEMO') {
            $notification = trans('user_validation.This Is Demo Version. You Can Not Change Anything');
            return response()->json(['message' => $notification], 403);
        }

        $user = Auth::guard('api')->user();
        if (!$user) {
            return response()->json(['message' => 'User Not Found'], 403);
        }

        $service_package = ListingPackage::find($service_package_id);

        if (!$service_package) {
            return response()->json([
                'message' => trans('Package Not Found')
            ], 403);
        }

        if ($package_name == 'Basic' || $package_name == 'Standard' || $package_name == 'Premium') {

            $auth_user = Session::get('auth_user');

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->first();

            if (!$service) {
                return response()->json([
                    'message' => trans('Service Not Found')
                ], 403);
            }

            $payable_amount = 0.00;
            if ($package_name == 'Basic') {
                $payable_amount = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $payable_amount = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $payable_amount = $service_package->premium_price;
            }

            $flutterwave_currency = Currency::findOrFail($this->payment_setting->flutterwave_currency_id);
        } else {
            $notify_message = trans('translate.Something went wrong');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }

        Session::put('auth_user', $user);
        $payment_setting = $this->payment_setting;
        $setting = GlobalSetting::first();


        return view('webview.flutterwave_webview', compact('user', 'payable_amount', 'setting', 'payment_setting', 'service_package', 'package_name', 'flutterwave_currency'));
    }


    public function flutterwave_webview_payment(Request $request, $service_package_id, $package_name)
    {

        $curl = curl_init();
        $tnx_id = $request->tnx_id;
        $url = "https://api.flutterwave.com/v3/transactions/$tnx_id/verify";
        $token = $this->payment_setting->flutterwave_secret_key;
        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => array(
                "Content-Type: application/json",
                "Authorization: Bearer $token"
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);
        $response = json_decode($response);

        if ($response->status == 'success') {

            $auth_user = Session::get('auth_user');

            $service_package = ListingPackage::findOrFail($service_package_id);

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

            $package_main_price = 0.00;
            if ($package_name == 'Basic') {
                $package_main_price = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $package_main_price = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $package_main_price = $service_package->premium_price;
            }

            $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Flutterwave', 'success', $tnx_id);

            $notify_message = trans('translate.Your payment has been made successful. Thanks for your new purchase');
            return response()->json(['status' => 'success', 'message' => $notify_message]);
        } else {
            $notify_message = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notify_message]);
        }
    }

    public function mollie_webview(Request $request, $service_package_id, $package_name)
    {

        if (env('APP_MODE') == 'DEMO') {
            $notification = trans('user_validation.This Is Demo Version. You Can Not Change Anything');
            return response()->json(['message' => $notification], 403);
        }

        $auth_user = Auth::guard('api')->user();
        if (!$auth_user) {
            return response()->json(['message' => 'User Not Found'], 403);
        }

        $service_package = ListingPackage::findOrFail($service_package_id);
        $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

        $package_main_price = 0.00;
        if ($package_name == 'Basic') {
            $package_main_price = $service_package->basic_price;
        } elseif ($package_name == 'Standard') {
            $package_main_price = $service_package->standard_price;
        } elseif ($package_name == 'Premium') {
            $package_main_price = $service_package->premium_price;
        }


        try {
            $mollie_currency = Currency::findOrFail($this->payment_setting->mollie_currency_id);

            $price = $package_main_price * $mollie_currency->currency_rate;
            $price = sprintf('%0.2f', $price);

            $mollie_api_key = $this->payment_setting->mollie_key;

            $currency = strtoupper($mollie_currency->currency_code);

            Mollie::api()->setApiKey($mollie_api_key);

            $payment = Mollie::api()->payments()->create([
                'amount' => [
                    'currency' => $currency,
                    'value' => '' . $price . '',
                ],
                'description' => env('APP_NAME'),
                'redirectUrl' => route('payment.mollie-webview-payment'),
            ]);

            $payment = Mollie::api()->payments()->get($payment->id);

            Session::put('service_package_id', $service_package_id);
            Session::put('package_name', $package_name);
            Session::put('payment_id', $payment->id);
            Session::put('auth_user', $auth_user);

            return redirect($payment->getCheckoutUrl(), 303);
        } catch (Exception $e) {
            $notify_message = trans('translate.Please provide valid mollie api key');
            return response()->json(['message' => $notify_message], 403);
        }
    }

    public function mollie_webview_payment(Request $request)
    {
        $mollie_api_key = $this->payment_setting->mollie_key;
        Mollie::api()->setApiKey($mollie_api_key);
        $payment = Mollie::api()->payments->get(session()->get('payment_id'));
        if ($payment->isPaid()) {

            $service_package_id = Session::get('service_package_id');

            $package_name = Session::get('package_name');

            $auth_user =  Session::get('auth_user');

            $service_package = ListingPackage::findOrFail($service_package_id);

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

            $package_main_price = 0.00;
            if ($package_name == 'Basic') {
                $package_main_price = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $package_main_price = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $package_main_price = $service_package->premium_price;
            }

            $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Mollie', 'success', session()->get('payment_id'));

            return redirect()->route('webview-success-payment');
        } else {
            return redirect()->route('webview-faild-payment');
        }
    }

    public function paystack_webview(Request $request, $service_package_id, $package_name)
    {
        if (env('APP_MODE') == 'DEMO') {
            $notification = trans('user_validation.This Is Demo Version. You Can Not Change Anything');
            return response()->json(['message' => $notification], 403);
        }

        $user = Auth::guard('api')->user();
        if (!$user) {
            return response()->json(['message' => 'User Not Found'], 403);
        }

        $service_package = ListingPackage::find($service_package_id);

        if (!$service_package) {
            return response()->json([
                'message' => trans('Package Not Found')
            ], 403);
        }

        if ($package_name == 'Basic' || $package_name == 'Standard' || $package_name == 'Premium') {

            $auth_user = Session::get('auth_user');

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->first();

            if (!$service) {
                return response()->json([
                    'message' => trans('Service Not Found')
                ], 403);
            }

            $payable_amount = 0.00;
            if ($package_name == 'Basic') {
                $payable_amount = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $payable_amount = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $payable_amount = $service_package->premium_price;
            }

            $paystack_currency = Currency::findOrFail($this->payment_setting->paystack_currency_id);
        } else {
            $notify_message = trans('translate.Something went wrong');
            return response()->json([
                'message' => $notify_message
            ], 403);
        }

        Session::put('auth_user', $user);
        $payment_setting = $this->payment_setting;
        $setting = GlobalSetting::first();

        return view('webview.paystack_webview', compact('user', 'payable_amount', 'setting', 'payment_setting', 'service_package', 'package_name', 'paystack_currency'));
    }


    public function paystack_webview_payment(Request $request, $service_package_id, $package_name)
    {

        $reference = $request->reference;
        $transaction = $request->tnx_id;
        $secret_key = $this->payment_setting->paystack_secret_key;
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://api.paystack.co/transaction/verify/$reference",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => array(
                "Authorization: Bearer $secret_key",
                "Cache-Control: no-cache",
            ),
        ));
        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);
        $final_data = json_decode($response);
        if ($final_data->status == true) {

            $auth_user = Session::get('auth_user');

            $service_package = ListingPackage::findOrFail($service_package_id);

            $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

            $package_main_price = 0.00;
            if ($package_name == 'Basic') {
                $package_main_price = $service_package->basic_price;
            } elseif ($package_name == 'Standard') {
                $package_main_price = $service_package->standard_price;
            } elseif ($package_name == 'Premium') {
                $package_main_price = $service_package->premium_price;
            }

            $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Paystack', 'success', $transaction);

            $notification = trans('translate.Your payment has been made successful. Thanks for your new purchase');
            return response()->json(['status' => 'success', 'message' => $notification]);
        } else {
            $notification = trans('translate.Something went wrong, please try again');
            return response()->json(['status' => 'faild', 'message' => $notification]);
        }
    }



    public function instamojo_webview(Request $request, $service_package_id, $package_name)
    {
        if (env('APP_MODE') == 'DEMO') {
            $notification = trans('user_validation.This Is Demo Version. You Can Not Change Anything');
            return response()->json(['message' => $notification], 403);
        }

        $notify_message = trans('translate.Your payment has been made successful. Thanks for your new purchase');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return response()->json(['status' => 'success', 'message' => $notify_message]);
    }

    public function instamojo_webview_payment(Request $request)
    {


        $input = $request->all();

        $environment = $this->payment_setting->instamojo_account_mode;
        $api_key = $this->payment_setting->instamojo_api_key;
        $auth_token = $this->payment_setting->instamojo_auth_token;

        if ($environment == 'Sandbox') {
            $url = 'https://test.instamojo.com/api/1.1/';
        } else {
            $url = 'https://www.instamojo.com/api/1.1/';
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url . 'payments/' . $request->get('payment_id'));
        curl_setopt($ch, CURLOPT_HEADER, FALSE);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);
        curl_setopt(
            $ch,
            CURLOPT_HTTPHEADER,
            array(
                "X-Api-Key:$api_key",
                "X-Auth-Token:$auth_token"
            )
        );
        $response = curl_exec($ch);
        $err = curl_error($ch);
        curl_close($ch);

        if ($err) {
            $service_package_id = Session::get('service_package_id');

            $package_name = Session::get('package_name');

            $notify_message = trans('translate.Something went wrong, please try again');
            $notify_message = array('message' => $notify_message, 'alert-type' => 'error');
            return redirect()->route('payment.pay', ['service_package_id' => $service_package_id, 'package_name' => $package_name])->with($notify_message);
        } else {
            $data = json_decode($response);
        }

        if ($data->success == true) {
            if ($data->payment->status == 'Credit') {

                $service_package_id = Session::get('service_package_id');

                $package_name = Session::get('package_name');

                $auth_user = Session::get('auth_user');

                $service_package = ListingPackage::findOrFail($service_package_id);

                $service = Listing::where(['status' => 'enable', 'approved_by_admin' => 'approved', 'id' => $service_package->listing_id])->firstOrFail();

                $package_main_price = 0.00;
                if ($package_name == 'Basic') {
                    $package_main_price = $service_package->basic_price;
                } elseif ($package_name == 'Standard') {
                    $package_main_price = $service_package->standard_price;
                } elseif ($package_name == 'Premium') {
                    $package_main_price = $service_package->premium_price;
                }

                $order = $this->create_order($auth_user, $service, $service_package, $package_name, $package_main_price, 'Instamojo', 'success', $request->get('payment_id'));

                return redirect()->route('webview-success-payment');
            }
        } else {
            return redirect()->route('webview-faild-payment');
        }
    }

    public function create_order($user, $service, $service_package, $package_name, $package_main_price, $payment_method, $payment_status, $transaction_id)
    {


        $order = new Order();
        $order->order_id = substr(rand(0, time()), 0, 10);
        $order->seller_id = $service->seller_id;
        $order->buyer_id = $user->id;
        $order->listing_id = $service->id;
        $order->listing_package_id = $service_package->id;
        $order->total_amount = $package_main_price;
        $order->package_amount = $package_main_price;
        $order->payment_method = $payment_method;
        $order->payment_status = $payment_status;
        $order->transaction_id = $transaction_id;

        if ($package_name == 'Basic') {

            $order->package_name = $service_package->basic_name;
            $order->package_description = $service_package->basic_description;
            $order->delivery_date = $service_package->basic_delivery_date;
            $order->revision = $service_package->basic_revision;
            $order->fn_website = $service_package->basic_fn_website;
            $order->number_of_page = $service_package->basic_page;
            $order->responsive = $service_package->basic_responsive;
            $order->source_code = $service_package->basic_source_code;
            $order->content_upload = $service_package->basic_content_upload;
            $order->speed_optimized = $service_package->basic_speed_optimized;
        } elseif ($package_name == 'Standard') {
            $order->package_name = $service_package->standard_name;
            $order->package_description = $service_package->standard_description;
            $order->delivery_date = $service_package->standard_delivery_date;
            $order->revision = $service_package->standard_revision;
            $order->fn_website = $service_package->standard_fn_website;
            $order->number_of_page = $service_package->standard_page;
            $order->responsive = $service_package->standard_responsive;
            $order->source_code = $service_package->standard_source_code;
            $order->content_upload = $service_package->standard_content_upload;
            $order->speed_optimized = $service_package->standard_speed_optimized;
        } elseif ($package_name == 'Premium') {
            $order->package_name = $service_package->premium_name;
            $order->package_description = $service_package->premium_description;
            $order->delivery_date = $service_package->premium_delivery_date;
            $order->revision = $service_package->premium_revision;
            $order->fn_website = $service_package->premium_fn_website;
            $order->number_of_page = $service_package->premium_page;
            $order->responsive = $service_package->premium_responsive;
            $order->source_code = $service_package->premium_source_code;
            $order->content_upload = $service_package->premium_content_upload;
            $order->speed_optimized = $service_package->premium_speed_optimized;
        }

        $order->save();


        return $order;
    }
}
