<?php

namespace App\Http\Controllers\API\Auth;

use Exception;
use Mail, Str;
use App\Models\User;
use App\Rules\Captcha;
use App\Helper\EmailHelper;
use Illuminate\Http\Request;
use App\Mail\UserRegistration;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Modules\EmailSetting\App\Models\EmailTemplate;
use Modules\SellerLevel\Entities\SellerLevel;
use Modules\SellerLevel\Entities\SellerLevelHistory;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest:api');
    }

    public function store_register(Request $request){

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', 'min:4', 'max:100'],
            'g-recaptcha-response'=>new Captcha()

        ],[
            'name.required' => trans('translate.Name is required'),
            'email.required' => trans('translate.Email is required'),
            'email.unique' => trans('translate.Email already exist'),
            'password.required' => trans('translate.Password is required'),
            'password.confirmed' => trans('translate.Confirm password does not match'),
            'password.min' => trans('translate.You have to provide minimum 4 character password'),
        ]);

        $is_seller = 0;
        if($request->user_type == 'seller'){
            $is_seller = 1;
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'username' => Str::slug($request->name).'-'.date('Ymdhis'),
            'status' => 'enable',
            'is_seller' => $is_seller,
            'is_banned' => 'no',
            'password' => Hash::make($request->password),
            'verification_otp' => random_int(100000, 999999)
        ]);

        if($user->is_seller == 1){
            $json_module_data = file_get_contents(base_path('modules_statuses.json'));
            $module_status = json_decode($json_module_data);

            if (property_exists($module_status, 'SellerLevel') && $module_status->SellerLevel === true) {
                $level = SellerLevel::findOrFail(1);

                SellerLevelHistory::create([
                    'seller_id' => $user->id,
                    'seller_level_id' => $level->id,
                    'level' => $level->name,
                    'assigned_at' => now(),
                    'orders_at_assignment' => 0,
                    'order_amount_at_assignment' => 0,
                ]);

                User::where('id', $user->id)->update(['current_level_id' => $level->id, 'level_assigned_at' => now()]);
            }
        }

        EmailHelper::mail_setup();

        try{
            $template = EmailTemplate::where('template_type', 'user_register_otp')->first();
            if($template){
                $subject=$template->subject;
                $message=$template->description;
                $message = str_replace('{{user_name}}',$request->name,$message);
                $message = str_replace('{{varification_otp}}',$user->verification_otp,$message);

                Mail::to($user->email)->send(new UserRegistration($message,$subject,$user));
            }

        }catch(Exception $ex){
            Log::info($ex->getMessage());
        }

        $notify_message = trans('translate.Account created successful, a verification OTP has been send to your mail, please verify it');

        return response()->json([
            'message' => $notify_message
        ]);


    }

    public function seller_store_register(Request $request){

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', 'min:4', 'max:100'],
            'g-recaptcha-response'=>new Captcha()

        ],[
            'name.required' => trans('translate.Name is required'),
            'email.required' => trans('translate.Email is required'),
            'email.unique' => trans('translate.Email already exist'),
            'password.required' => trans('translate.Password is required'),
            'password.confirmed' => trans('translate.Confirm password does not match'),
            'password.min' => trans('translate.You have to provide minimum 4 character password'),
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'username' => Str::slug($request->name).'-'.date('Ymdhis'),
            'status' => 'enable',
            'is_banned' => 'no',
            'is_seller' => 1,
            'password' => Hash::make($request->password),
            'verification_otp' => random_int(100000, 999999)
        ]);

        EmailHelper::mail_setup();

        try{
            $template = EmailTemplate::where('template_type', 'user_register_otp')->first();
            if($template){
                $subject=$template->subject;
                $message=$template->description;
                $message = str_replace('{{user_name}}',$request->name,$message);
                $message = str_replace('{{varification_otp}}',$user->verification_otp,$message);

                Mail::to($user->email)->send(new UserRegistration($message,$subject,$user));
            }

        }catch(Exception $ex){
            Log::info($ex->getMessage());
        }

        $notify_message = trans('translate.Account created successful, a verification OTP has been send to your mail, please verify it');

        return response()->json([
            'message' => $notify_message
        ]);

    }

    public function resend_register_code(Request $request){

        $rules = [
            'email'=>'required',
        ];
        $customMessages = [
            'email.required' => trans('translate.Email is required'),
        ];
        $this->validate($request, $rules,$customMessages);

        $user = User::where('email', $request->email)->first();
        if($user){
            if($user->email_verified_at == null){
                try{
                    EmailHelper::mail_setup();

                    $template = EmailTemplate::where('template_type', 'user_register_otp')->first();
                    if($template){
                        $subject=$template->subject;
                        $message=$template->description;
                        $message = str_replace('{{user_name}}',$user->name,$message);
                        $message = str_replace('{{varification_otp}}',$user->verification_otp,$message);

                        Mail::to($user->email)->send(new UserRegistration($message,$subject,$user));
                    }

                }catch(Exception $ex){
                    Log::info($ex->getMessage());
                }

                $notification = trans('translate.OTP resend successfully');
                return response()->json(['message' => $notification]);
            }else{
                $notification = trans('translate.Email already verified');
                return response()->json(['message' => $notification],403);
            }

        }else{
            $notification = trans('translate.Email does not exist');
            return response()->json(['message' => $notification],403);
        }
    }


    public function register_verification(Request $request){

        $rules = [
            'email'=>'required',
            'otp'=>'required',
        ];
        $customMessages = [
            'email.required' => trans('translate.Email is required'),
        ];
        $this->validate($request, $rules,$customMessages);

        $user = User::where('verification_otp',$request->otp)->where('email', $request->email)->first();
        if($user){

            if($user->email_verified_at != null){
                $notify_message = trans('translate.Email already verified');
                return response()->json(['message' => $notify_message],403);
            }

            $user->email_verified_at = date('Y-m-d H:i:s');
            $user->verification_token = null;
            $user->save();

            $notify_message = trans('translate.Verification Successfully');
            return response()->json(['message' => $notify_message]);
        }else{

            $notify_message = trans('translate.Invalid token or email');
            return response()->json(['message' => $notify_message],403);
        }
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
        ]);
    }
}
