<?php

namespace Modules\SellerLevel\Http\Controllers;

use App\Models\User;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Modules\SellerLevel\Entities\SellerLevel;
use Modules\SellerLevel\Entities\SellerLevelHistory;
use Image, Str, File;

class SellerLevelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $levels = SellerLevel::all();

        return view('sellerlevel::index', [
            'levels' => $levels
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('sellerlevel::create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validation
        $request->validate([
            'name' => 'required|string|max:255',
            'min_orders' => 'required|integer|min:0',
            'commission' => 'required|numeric|min:0|max:100',
            'basic_price' => 'required|numeric|min:0',
            'min_order_amount' => 'required|numeric|min:0',
            'icon' => 'required',
        ], [
            'name.required' => trans('translate.Level name is required'),
            'min_orders.required' => trans('translate.Minimum orders are required'),
            'commission.required' => trans('translate.Commission is required'),
            'basic_price.required' => trans('translate.Basic package price is required'),
            'min_order_amount.required' => trans('translate.Minimum order amount is required'),
            'icon.required' => trans('translate.Icon is required'),
        ]);

        // Create new Seller Level
        $sellerLevel = new SellerLevel();


        if($request->file('icon')){
            $icon_image = $request->icon;
            $extention = $icon_image->getClientOriginalExtension();
            $image_name = 'selle-level'.date('-Y-m-d-h-i-s-').rand(999,9999).'.'.$extention;
            $image_name = 'uploads/custom-images/'.$image_name;
            Image::make($icon_image)->save(public_path().'/'.$image_name);
            $sellerLevel->icon = $image_name;
            
        }


        $sellerLevel->name = $request->name;
        $sellerLevel->min_orders = $request->min_orders;
        $sellerLevel->commission = $request->commission;
        $sellerLevel->basic_price = $request->basic_price;
        $sellerLevel->min_order_amount = $request->min_order_amount;
        $sellerLevel->save();

        $notification = trans('translate.Create Successfully');
        $notification = array('message'=>$notification,'alert-type'=>'success');
        return redirect()->route('admin.sellerlevel.index')->with($notification);

    }

    /**
     * Show the specified resource.
     */
    public function show($id)
    {
        return view('sellerlevel::show');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {

        $level = SellerLevel::findOrFail($id);

        return view('sellerlevel::edit', [
            'level' => $level
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $sellerLevel = SellerLevel::findOrFail($id);

        if($request->file('icon')){
            $old_icon = $sellerLevel->icon;
            $icon_image = $request->icon;
            $extention = $icon_image->getClientOriginalExtension();
            $image_name = 'selle-level'.date('-Y-m-d-h-i-s-').rand(999,9999).'.'.$extention;
            $image_name = 'uploads/custom-images/'.$image_name;
            Image::make($icon_image)->save(public_path().'/'.$image_name);
            $sellerLevel->icon = $image_name;
            $sellerLevel->save();

            if($old_icon){
                if(File::exists(public_path().'/'.$old_icon))unlink(public_path().'/'.$old_icon);
            }
            
        }

        $sellerLevel->name = $request->name;
        $sellerLevel->min_orders = $request->min_orders;
        $sellerLevel->commission = $request->commission;
        $sellerLevel->basic_price = $request->basic_price;
        $sellerLevel->min_order_amount = $request->min_order_amount;
        $sellerLevel->save();

        $notification = trans('translate.Updated Successfully');
        $notification = array('message'=>$notification,'alert-type'=>'success');
        return redirect()->route('admin.sellerlevel.index')->with($notification);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {

        if($id == 1){
            $notification = trans('translate.This one default seller level, so you can not delete it');
            $notification = array('message'=>$notification,'alert-type'=>'error');
            return redirect()->route('admin.sellerlevel.index')->with($notification);
        }

        $sellerLevel = SellerLevel::findOrFail($id);

        $seller_history = SellerLevelHistory::where('seller_level_id', $id)->count();
        if($seller_history > 0){
            $notification = trans('translate.This one already assinged to seller, so you can not delete it');
            $notification = array('message'=>$notification,'alert-type'=>'error');
            return redirect()->route('admin.sellerlevel.index')->with($notification);
        }

        $old_icon = $sellerLevel->icon;
        if($old_icon){
            if(File::exists(public_path().'/'.$old_icon))unlink(public_path().'/'.$old_icon);
        }

        $sellerLevel->delete();

        $notification = trans('translate.Delete Successfully');
        $notification = array('message'=>$notification,'alert-type'=>'success');
        return redirect()->route('admin.sellerlevel.index')->with($notification);

    }

    
    public function assign_sellerlevel(Request $request){

        abort(404);

        $levels = SellerLevel::where('id', '!=', 1)->get();

        $sellers = [];

        if ($request->has('is_search') && $request->has('level_id') && $request->level_id != null) {
            $level = SellerLevel::findOrFail($request->level_id);

            $assignment_method = env('SELLER_LEVEL_ASSIGNMENT_METHOD', 'registration_date'); // Default to registration_date if not set

            $sellersQuery = User::where('is_influencer', 'yes')
                ->where('status', 'enable')
                ->orderBy('id', 'desc');

            $sellers = $sellersQuery->get();

            $eligible_sellers = collect();

            foreach ($sellers as $seller) {
                $lastAssignment = SellerLevelHistory::where('seller_id', $seller->id)
                    ->orderBy('assigned_at', 'desc')
                    ->first();

                $previousAssignments = SellerLevelHistory::where('seller_id', $seller->id)
                ->where('seller_level_id', $level->id)
                ->exists();

                if ($previousAssignments) {
                    continue;
                }

                if ($lastAssignment && $assignment_method === 'level_assignment') {
                    $startDate = $lastAssignment->assigned_at;
                } else {
                    $startDate = $seller->created_at;
                }

                $seller->total_orders = Order::where('influencer_id', $seller->id)
                    ->where('order_status', 'complete')
                    ->where('created_at', '>=', $startDate)
                    ->count();

                $seller->total_order_amount = Order::where('influencer_id', $seller->id)
                    ->where('order_status', 'complete')
                    ->where('created_at', '>=', $startDate)
                    ->sum('total_amount');


                if ($seller->total_orders >= $level->min_orders && $seller->total_order_amount >= $level->min_order_amount) {
                    $eligible_sellers->push($seller);
                }
            }

            if ($eligible_sellers->isEmpty()) {
                $notification = trans('translate.No eligible sellers for this level. All sellers have already been assigned this level or don’t meet the criteria.');
                $notification = array('message' => $notification, 'alert-type' => 'warning');
                return redirect()->route('admin.assign-sellerlevel')->with($notification);
            }

            $sellers = $eligible_sellers;

        }

        return view('sellerlevel::assign_sellerlevel', [
            'levels' => $levels,
            'sellers' => $sellers,
        ]);
    }

    public function store_assign_sellerlevel(Request $request, $level_id)
    {

        abort(404);
        
        $level = SellerLevel::findOrFail($level_id);

        $assignment_method = env('SELLER_LEVEL_ASSIGNMENT_METHOD', 'registration_date'); 

        $sellersQuery = User::where('is_influencer', 'yes')
            ->where('status', 'enable')
            ->orderBy('id', 'desc');

        $sellers = $sellersQuery->get();

        $eligible_sellers = collect();
 
        foreach ($sellers as $seller) {
            $lastAssignment = SellerLevelHistory::where('seller_id', $seller->id)
                ->orderBy('assigned_at', 'desc')
                ->first();

            if ($assignment_method === 'level_assignment' && $lastAssignment) {
                $startDate = $lastAssignment->assigned_at;
            } else {
                $startDate = $seller->created_at;
            }

            $seller->total_orders = Order::where('influencer_id', $seller->id)
                ->where('order_status', 'complete')
                ->where('created_at', '>=', $startDate)
                ->count();

            $seller->total_order_amount = Order::where('influencer_id', $seller->id)
                ->where('order_status', 'complete')
                ->where('created_at', '>=', $startDate)
                ->sum('total_amount');

            if ($lastAssignment && $lastAssignment->seller_level_id == $level->id) {
                continue;
            }

            if ($seller->total_orders >= $level->min_orders && $seller->total_order_amount >= $level->min_order_amount) {
                $eligible_sellers->push($seller);
            }
        }

        if ($eligible_sellers->isEmpty()) {
            $notification = trans('translate.No eligible sellers for this level. All sellers have already been assigned this level.');
            $notification = array('message' => $notification, 'alert-type' => 'warning');
            return redirect()->route('admin.assign-sellerlevel')->with($notification);
        }

        $seller_ids = [];
        foreach ($eligible_sellers as $seller) {
            SellerLevelHistory::create([
                'seller_id' => $seller->id,
                'seller_level_id' => $level->id,
                'level' => $level->name,
                'assigned_at' => now(),
                'orders_at_assignment' => $seller->total_orders,
                'order_amount_at_assignment' => $seller->total_order_amount,
            ]);

            $seller_ids[] = $seller->id;
        }

        User::whereIn('id', $seller_ids)->update(['current_level_id' => $level->id, 'level_assigned_at' => now()]);


        $notification = trans('translate.Level assigned successful');
        $notification = array('message' => $notification, 'alert-type' => 'success');
        return redirect()->route('admin.assign-sellerlevel')->with($notification);
    }


 
    
}
