<?php

namespace Modules\AdminTeam\Database\Seeders;

use App\Models\Admin;
use Illuminate\Database\Seeder;
use Modules\AdminTeam\App\Models\AdminRole;
use Modules\AdminTeam\App\Models\AdminPermissionList;

class AdminRoleSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $roles = [
            [
                'name' => 'super_admin',
                'display_name' => 'Super Administrator',
                'description' => 'Full system access and control. Can manage all aspects of the system including other admins, roles, and permissions.',
                'status' => 'active',
                'is_system_role' => true,
            ],
            [
                'name' => 'admin',
                'display_name' => 'Administrator',
                'description' => 'General administrator with access to most system functions except admin management.',
                'status' => 'active',
                'is_system_role' => false,
            ],
            [
                'name' => 'content_manager',
                'display_name' => 'Content Manager',
                'description' => 'Manages content including blogs, pages, and frontend sections.',
                'status' => 'active',
                'is_system_role' => false,
            ],
            [
                'name' => 'service_manager',
                'display_name' => 'Service Manager',
                'description' => 'Manages services, categories, and service-related operations.',
                'status' => 'active',
                'is_system_role' => false,
            ],
            [
                'name' => 'order_manager',
                'display_name' => 'Order Manager',
                'description' => 'Manages orders and order-related operations.',
                'status' => 'active',
                'is_system_role' => false,
            ],
            [
                'name' => 'user_manager',
                'display_name' => 'User Manager',
                'description' => 'Manages sellers, buyers, and user-related operations.',
                'status' => 'active',
                'is_system_role' => false,
            ],
            [
                'name' => 'viewer',
                'display_name' => 'Viewer',
                'description' => 'Read-only access to view data and reports.',
                'status' => 'active',
                'is_system_role' => false,
            ],
        ];

        foreach ($roles as $role) {
            AdminRole::updateOrCreate(
                ['name' => $role['name']],
                $role
            );
        }

        // Assign permissions to roles
        $this->assignPermissionsToRoles();

        // Assign super admin role to existing super admin user
        $admin = Admin::where('admin_type', 'super_admin')->first();
        if ($admin) {
            $superAdminRole = AdminRole::where('name', 'super_admin')->first();
            if ($superAdminRole) {
                $admin->roles()->sync([$superAdminRole->id]);
            }
        }
    }

    private function assignPermissionsToRoles()
    {
        // Get all permissions
        $allPermissions = AdminPermissionList::where('is_group', false)->pluck('id')->toArray();

        // Super Admin - All permissions
        $superAdminRole = AdminRole::where('name', 'super_admin')->first();
        if ($superAdminRole) {
            $superAdminRole->permissions()->sync($allPermissions);
        }

        // Admin - Most permissions except admin team management
        $adminRole = AdminRole::where('name', 'admin')->first();
        if ($adminRole) {
            $adminPermissions = AdminPermissionList::where('is_group', false)
                ->where('module', '!=', 'admin_team')
                ->pluck('id')
                ->toArray();
            $adminRole->permissions()->sync($adminPermissions);
        }

        // Content Manager - CMS, Pages, and Section permissions
        $contentManagerRole = AdminRole::where('name', 'content_manager')->first();
        if ($contentManagerRole) {
            $contentPermissions = AdminPermissionList::where('is_group', false)
                ->whereIn('module', ['cms', 'pages', 'section'])
                ->pluck('id')
                ->toArray();
            $contentManagerRole->permissions()->sync($contentPermissions);
        }

        // Service Manager - Service and related permissions
        $serviceManagerRole = AdminRole::where('name', 'service_manager')->first();
        if ($serviceManagerRole) {
            $servicePermissions = AdminPermissionList::where('is_group', false)
                ->whereIn('module', ['service', 'service_reports'])
                ->pluck('id')
                ->toArray();
            $serviceManagerRole->permissions()->sync($servicePermissions);
        }

        // Order Manager - Order and withdraw permissions
        $orderManagerRole = AdminRole::where('name', 'order_manager')->first();
        if ($orderManagerRole) {
            $orderPermissions = AdminPermissionList::where('is_group', false)
                ->whereIn('module', ['order', 'withdraw'])
                ->pluck('id')
                ->toArray();
            $orderManagerRole->permissions()->sync($orderPermissions);
        }

        // User Manager - Seller, Buyer, and Contact permissions
        $userManagerRole = AdminRole::where('name', 'user_manager')->first();
        if ($userManagerRole) {
            $userPermissions = AdminPermissionList::where('is_group', false)
                ->whereIn('module', ['seller', 'buyer', 'contact'])
                ->pluck('id')
                ->toArray();
            $userManagerRole->permissions()->sync($userPermissions);
        }

        // Viewer - Read-only permissions
        $viewerRole = AdminRole::where('name', 'viewer')->first();
        if ($viewerRole) {
            $viewPermissions = AdminPermissionList::where('is_group', false)
                ->whereIn('module', ['dashboard', 'order', 'service', 'jobpost', 'seller', 'buyer', 'contact', 'service_reports', 'cms', 'pages', 'section'])
                ->pluck('id')
                ->toArray();
            $viewerRole->permissions()->sync($viewPermissions);
        }
    }
}
