<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use App\Models\Order;
use App\Helper\EmailHelper;
use Illuminate\Http\Request;
use App\Mail\SendSellerLevelMail;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Mail;
use Modules\JobPost\Entities\JobRequest;
use Modules\SellerLevel\Entities\SellerLevel;
use Modules\EmailSetting\App\Models\EmailTemplate;
use Modules\SellerLevel\Entities\SellerLevelHistory;

class OrderController extends Controller
{
    public function index(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->latest()->get();


        return view('admin.order_list', [
            'orders' => $orders
        ]);
    }


    public function active_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_active_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where(['approved_by_seller' => 'approved', 'order_status' => 'approved_by_seller'])->latest()->get();

        return view('admin.order_list', [
            'orders' => $orders
        ]);
    }

    public function awaiting_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_awaiting_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where(['approved_by_seller' => 'pending'])->where('order_status' , '!=', 'cancel_by_buyer')->latest()->get();

        return view('admin.awaiting_orders', [
            'orders' => $orders
        ]);
    }

    public function reject_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_rejected_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where(['approved_by_seller' => 'rejected'])->where('order_status' , '!=', 'cancel_by_buyer')->latest()->get();

        return view('admin.reject_orders', [
            'orders' => $orders
        ]);
    }

    public function cancel_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_cancel_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where(function ($query) {
            $query->where('order_status', 'cancel_by_seller')
                  ->orWhere('order_status', 'cancel_by_buyer');
        })->latest()->get();

        return view('admin.cancel_orders', [
            'orders' => $orders
        ]);
    }

    public function complete_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_complete_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where(function ($query) {
            $query->where('order_status', 'complete_by_buyer')
                  ->orWhere('completed_by_buyer', 'complete');
        })->latest()->get();

        return view('admin.complete_orders', [
            'orders' => $orders
        ]);
    }

    public function pending_payment_orders(){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_pending_payment_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $orders = Order::with('listing', 'seller', 'buyer')->where('payment_status', 'pending')->latest()->get();

        return view('admin.pending_payment_orders', [
            'orders' => $orders
        ]);
    }

    public function order_show($order_id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_detail_view')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::with('listing')->where('order_id', $order_id)->first();

        $seller = User::findOrFail($order->seller_id);

        $total_job = JobRequest::where('seller_id', $seller->id)->where('status', 'approved')->count();

        return view('admin.order_show', [
            'order' => $order,
            'seller' => $seller,
            'total_job' => $total_job,
        ]);
    }


    public function order_cancel(Request $request, $id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_cancel')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::where('id', $id)->first();
        $order->order_status = 'cancel_by_buyer';
        $order->save();

        $notify_message = trans('translate.Order cancel successful');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->back()->with($notify_message);

    }

    public function order_approved(Request $request, $id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_approve')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::where('id', $id)->first();
        $order->approved_by_seller = 'approved';
        $order->order_status = 'approved_by_seller';
        $order->save();

        $notify_message = trans('translate.Order approval successful');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->back()->with($notify_message);

    }


    public function order_complete(Request $request, $id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_approve')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::where('id', $id)->first();
        $order->completed_by_buyer = 'complete';
        $order->order_status = 'complete_by_buyer';
        $order->save();

        $json_module_data = file_get_contents(base_path('modules_statuses.json'));
        $module_status = json_decode($json_module_data);
        
        if (property_exists($module_status, 'SellerLevel') && $module_status->SellerLevel === true) {

            $assignment_method = env('SELLER_LEVEL_ASSIGNMENT_METHOD', 'registration_date');

            $seller = User::findOrFail($order->seller_id);

            $eligible_sellers = collect();

            $lastAssignment = SellerLevelHistory::where('seller_id', $seller->id)
                    ->orderBy('assigned_at', 'desc')
                    ->first();

            if ($assignment_method === 'level_assignment' && $lastAssignment) {
                $startDate = $lastAssignment->assigned_at;
            } else {
                $startDate = $seller->created_at;
            }

            $seller->total_orders = Order::where('seller_id', $seller->id)
                ->where('order_status', 'complete_by_buyer')
                ->where('created_at', '>=', $startDate)
                ->count();

            $seller->total_order_amount = Order::where('seller_id', $seller->id)
                ->where('order_status', 'complete_by_buyer')
                ->where('created_at', '>=', $startDate)
                ->sum('total_amount');

            $level = SellerLevel::where('id', '>', $seller->current_level_id)->first();

            if($level){
                if ($lastAssignment && $lastAssignment->seller_level_id != $level->id) {
                    if ($seller->total_orders >= $level->min_orders && $seller->total_order_amount >= $level->min_order_amount) {

                        try{
                            EmailHelper::mail_setup();

                            $template = EmailTemplate::where('id',5)->first();
                            $subject = $template->subject;
                            $message=$template->description;

                            $message = str_replace('{{name}}',$seller->name,$message);
                            $message = str_replace('{{lev_name}}',$level->name,$message);

                            Mail::to($seller->email)->send(new SendSellerLevelMail($message,$subject));
                        }catch( \Exception $e ){
                        
                            Log::error('Mail send error:' . $e->getMessage());
                        }

                        SellerLevelHistory::create([
                            'seller_id' => $seller->id,
                            'seller_level_id' => $level->id,
                            'level' => $level->name,
                            'assigned_at' => now(),
                            'orders_at_assignment' => $seller->total_orders,
                            'order_amount_at_assignment' => $seller->total_order_amount,
                        ]);

                        User::where('id', $seller->id)->update(['current_level_id' => $level->id, 'level_assigned_at' => now()]);

                    }
                }
            }
        }

        $notify_message = trans('translate.Order complete successful');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->back()->with($notify_message);

    }

    public function payment_approval(Request $request, $id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_approve')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::where('id', $id)->first();
        $order->payment_status = 'success';
        $order->save();

        $notify_message = trans('translate.Payment approval successful');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->back()->with($notify_message);

    }

    public function order_delete(Request $request, $id){
        $admin = auth('admin')->user();
        if(!$admin->hasPermission('order_delete')){
            abort(403, 'You do not have permission to access this resource.');
        }
        $order = Order::where('id', $id)->first();
        $order->delete();

        $notify_message = trans('translate.Order delete successful');
        $notify_message = array('message' => $notify_message, 'alert-type' => 'success');
        return redirect()->route('admin.orders')->with($notify_message);

    }


}
